#include "config.h"
#include "judge.h"
#include "common.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <fcntl.h>
#include <dirent.h>
#include <unistd.h>
#include <errno.h>
#include <grp.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/resource.h>
#include <sys/time.h>
#include <sys/wait.h>

#include <pthread.h>
#include <semaphore.h>

#ifdef HAVE_LINUX_EXTENSIONS
#include <sys/prctl.h>
#include <sys/capability.h>
#endif

#define abort_makechild(val) \
	do { \
	sctjudge_makechild_cleanup(&mcopt, chrdir_oldmode, execdestname); \
	(*(int*)toreturn) = (val); \
	pthread_exit(toreturn); \
	}while(0)

#define SCTCHILD_OPEN_INPUT			 0
#define SCTCHILD_DUP2_INPUT			 1
#define SCTCHILD_OPEN_OUTPUT		 2
#define SCTCHILD_DUP2_OUTPUT		 3
#define SCTCHILD_OPEN_STDERR		 4
#define SCTCHILD_DUP2_STDERR		 5
#define SCTCHILD_CLOSE_OTHERFD		 6
#define SCTCHILD_SETRLIMIT_VM		 7
#define SCTCHILD_SETRLIMIT_FSIZE	 8
#define SCTCHILD_SETRLIMIT_OPENFILE	 9
#define SCTCHILD_SETRLIMIT_PROC		10
#define SCTCHILD_CHROOT				11
#define SCTCHILD_SETUID				12
#define SCTCHILD_SETGID				13
#define SCTCHILD_SETGROUPS			14
#define SCTCHILD_CAPSET				15
#define SCTCHILD_EXEC				16
#define SCTCHILD_MSGMAX				17 /* 訊息代碼的最大值 */
#define SCTCHILD_WRITE_SUCCMSG(reason) \
	do{ \
		childmsg[0] = (reason); \
		childmsg[1] = 0; \
		write(childpipe[1], (void*)childmsg, sizeof(int)*2);  \
	}while(0)
#define SCTCHILD_WRITE_FAILMSG(reason) \
	do{ \
		childmsg[0] = (reason); \
		childmsg[1] = errno; \
		write(childpipe[1], (void*)childmsg, sizeof(int)*2);  \
	}while(0)

#define SCTRES_OK	0
#define SCTRES_RE	1
#define SCTRES_TLE	2
#define SCTRES_OLE	3
#define SCTRES_SLE	4	/* 暫停次數太多 */
#define SCTRES_MAX	5

/* XXX 這種沒用的 macro 就丟了吧 */
#define getdvalue_quotient(num, div)	((num)/(div))
#define getdvalue_remainder(num, div) 	((num)%(div))

pid_t pidchild;
pthread_mutex_t pidmutex, tkill_mx, tdisplay_mx, judge_tle_mx;
pthread_t tkill, tdisplay;
char tkill_yes = 0, tdisplay_yes = 0, judge_tle = 0;
sem_t mcthr, addthr;

static const char* childmsg_text[SCTCHILD_MSGMAX] = {
	"開啟輸入檔案",
	"重新導向標準輸入",
	"開啟輸出檔案",
	"重新導向標準輸出",
	"開啟用於導向標準錯誤的檔案",
	"重新導向標準錯誤",
	"關閉所有不使用的檔案",
	"設定記憶體限制",
	"設定輸出限制",
	"設定禁止開啟其他檔案",
	"設定禁止產生新程序",
	"執行 chroot",
	"修改 real 和 effective UID",
	"修改 real 和 effective GID",
	"修改 supplementary GIDs",
	"丟棄所有 Linux capabilities",
	"執行受測程式"
};

/* UD (undefined) 並不會寫在這裡 */
const char* sctres_text[2][SCTRES_MAX] = {
	{"OK", "RE", "TLE", "OLE", "SLE"},
	{
		"正常結束",
		"執行過程中發生錯誤",
		"超過時間限制",
		"輸出超過限制",
		"暫停次數太多"
	}
};

static void sctjudge_makechild_cleanup(mcopt, oldperm, copiedexe)
	const struct makechildopt* mcopt;
	const mode_t oldperm;
	char* copiedexe;
{
	if(oldperm != -1 && mcopt->chrootdir != NULL){
		if(mcopt->flags & SCTMC_VERBOSE){
			printf("恢復 %s 的權限\n", mcopt->chrootdir);
		}
		chmod(mcopt->chrootdir, oldperm);
	}
	if(copiedexe != NULL && mcopt->chrootdir != NULL && 
			!(mcopt->flags & SCTMC_NOCOPY)){
		if(mcopt->flags & SCTMC_VERBOSE){
			printf("移除 %s......\n", copiedexe);
		}
		unlink(copiedexe);
		free(copiedexe);
	}

	pthread_mutex_lock(&tkill_mx);
	if(tkill_yes){
		tkill_yes = 0;
		pthread_mutex_unlock(&tkill_mx);
		pthread_cancel(tkill);
	}else{
		pthread_mutex_unlock(&tkill_mx);
	}

	pthread_mutex_lock(&tdisplay_mx);
	if(tdisplay_yes){
		tdisplay_yes = 0;
		pthread_mutex_unlock(&tdisplay_mx);
		pthread_cancel(tdisplay);
	}else{
		pthread_mutex_unlock(&tdisplay_mx);
	}
}

static int copyfilebyname(const char* src, const char* destdir
		, char** srcsn, char** destfn){
	/* strerror 並不 thread safe，因此必須確定只有一個 thread 會用到 */
	/* 為了確保安全，要求目的地目錄必須是空的 */
	DIR* dirp;
	if((dirp=opendir(destdir)) == NULL){
		fprintf(stderr, "無法開啟目錄 %s：%s\n", destdir, strerror(errno));
		return -1;
	}
	struct dirent* entryp;
	while((entryp=readdir(dirp)) != NULL){
		if(entryp->d_name[0] != '.' || 
			(entryp->d_name[1] != '.' && entryp->d_name[1] != '\0')){
			fprintf(stderr, "拒絕複製檔案：目錄 %s 不是空的\n", destdir);
			return -1;
		}
	}
	int fd, fd2;
	if((fd=open(src, O_RDONLY)) < 0){
		fprintf(stderr, "無法讀取檔案 %s：%s\n", src, strerror(errno));
		return -1;
	}
	const int bufsize = 4096;
	void* buf = malloc(bufsize);
	char* srcshortname;
	char* tmp;
	if((tmp=strrchr(src, '/')) == NULL){
		srcshortname = (char*)src;
	}else{
		srcshortname = tmp + 1;
	}
	*srcsn = srcshortname;
	char* destfilename = (void*)
		malloc(strlen(srcshortname) + strlen(destdir) + 1);
	sprintf(destfilename, "%s/%s", destdir, srcshortname);
	*destfn = destfilename;
	/* 新檔案權限是 0755，umask 亂弄的我可不管 */
	if((fd2=open(destfilename, O_CREAT | O_WRONLY | O_EXCL, 
				S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH)) < 0){
		fprintf(stderr, "無法建立檔案 %s：%s\n", 
				destfilename, strerror(errno));
		free(buf);
		return -1;
	}
	int readcount;
	while((readcount = read(fd, buf, bufsize)) > 0){
		write(fd2, buf, readcount);
	}
	close(fd);
	close(fd2);
	free(buf);
	return 0;
}

static int read_size(int fd, void* buf, size_t count){
	/* 一定要讀到 count 位元組的資料才會回傳，所以只要回傳 < count
	 * 就表示已經讀到 EOF 了 */
	char* bufp = (char*)buf;
	size_t curcount = 0;
	int rval;
	do{
		rval = read(fd, bufp, count - curcount);
		if(rval < 0){
			return rval;
		}else if(rval == 0){
			return curcount;
		}else{
			bufp += rval;
			curcount += rval;
		}
	}while(curcount < count);
	return count;
}

void* sctjudge_makechild(void* arg){
	sem_wait(&mcthr);
	sem_destroy(&mcthr);
	void* toreturn = malloc(sizeof(int));
	struct makechildopt mcopt = *(struct makechildopt*)arg;
	char* execshortname, *execdestname = NULL;
	/* 這幾個變數用來儲存 Judge 結果，如果沒有啟用 verbose 
	 * 且一切正常，那這些是唯一會輸出的東西了，目的是輸出容
	 * 易被其他程式分析 */
	int judge_result;		/* 就是那個兩三個英文字母的代碼 */
	int judge_exitcode; 	/* 程式結束回傳值 */
	struct timespec judge_time;			/* 執行時間 */
	int judge_signal;		/* RE 時的訊號 */

	struct stat chrdirinfo;
	mode_t chrdir_oldmode = -1;

	if(mcopt.flags & SCTMC_VERBOSE){
		puts("正在檢查設定值是否正確......");
	}
	if(mcopt.executable == NULL){
		fputs("受測程式可執行檔名稱為必要參數，不可為空白\n", stderr);
		abort_makechild(1);
	}
	if(mcopt.inputfile == NULL){
		mcopt.inputfile = NULL_DEVICE;
	}
	if(mcopt.outputfile == NULL){
		fputs("輸出檔案名稱必要參數，不可為空白\n", stderr);
		abort_makechild(1);
	}
	if(mcopt.exectime <= 0){
		fputs("執行時間限制為必要參數，不可為空白！\n", stderr);
		abort_makechild(1);
	}
	if(mcopt.flags & SCTMC_VERBOSE){
		puts("列出設定值：");
		printf("\t受測程式可執行檔名稱：%s", mcopt.executable);
		if(mcopt.flags & SCTMC_NOCOPY && mcopt.chrootdir != NULL){
			printf(" (相對於 %s)\n"
				"\t受測程式執行時的根目錄：%s\n"
				"\t執行前複製檔案：否\n"
				, mcopt.chrootdir, mcopt.chrootdir);
		}else{
			putchar('\n');
			if(mcopt.chrootdir != NULL){
				printf("\t受測程式執行時的根目錄：%s\n"
					"\t執行前複製檔案：是\n"
					, mcopt.chrootdir);
			}else{
				puts("\t受測程式執行時的根目錄：/\n"
				"\t執行前複製檔案：否");
			}
		}
		printf("\t輸入檔案：%s\n"
			"\t輸出檔案：%s"
			, mcopt.inputfile, mcopt.outputfile);
		if(mcopt.flags & SCTMC_REDIR_STDERR){
			puts(" (重新導向標準輸出和標準錯誤)");
		}else{
			puts(" (重新導向標準輸出)");
		}
		printf("\t執行時間限制：%d 毫秒\n", mcopt.exectime);
		fputs("\t記憶體使用量限制：", stdout);
		if(mcopt.memlimit > 0){
			printf("%d MiB\n", mcopt.memlimit);
		}else{
			puts("無限制");
		}
		fputs("\t輸出限制：", stdout);
		if(mcopt.outlimit > 0){
			printf("%d MiB\n", mcopt.outlimit);
		}else{
			puts("無限制");
		}
		if(mcopt.flags & SCTMC_SETUID){
			printf("\t設定受測程式執行時的 UID 為 %d\n", mcopt.uid);
		}else{
			puts("\t執行受測程式時維持原有的 real UID 和 effective UID");
		}
		if(mcopt.flags & SCTMC_SETGID){
			printf("\t設定受測程式執行時的 GID 為 %d\n", mcopt.gid);
		}else{
			puts("\t執行受測程式時維持原有的 real GID、effective GID "
					"和 supplementary GID");
		}
	}
	if(mcopt.flags & SCTMC_DRYRUN){
		abort_makechild(0);
	}
	/* 我們要開始做正事了 */
	/* 由此開始，我是唯一會用到 strerror() 的 thread
	 * 所以說應該沒有問題 */
	if(mcopt.chrootdir != NULL && !(mcopt.flags & SCTMC_NOCOPY)){
		/* 必須要複製可執行檔 */
		if(mcopt.flags & SCTMC_VERBOSE){
			puts("開始複製可執行檔......");
		}
		if(copyfilebyname(mcopt.executable, mcopt.chrootdir, &execshortname
					, &execdestname) < 0){
			abort_makechild(2);
		}
	}
	/* 再來是 chroot 的問題，避免受測程式亂開目錄之類的，
	 * 所以把權限設成 0555，不過如果沒有變更 uid 和 gid ，
	 * 受測程式還是可以 chmod 回來 */
	if(mcopt.chrootdir != NULL){
		if(mcopt.flags & SCTMC_VERBOSE){
			puts("取得用於 chroot 的目錄相關資訊");
		}
		if(stat(mcopt.chrootdir, &chrdirinfo) < 0){
			fprintf(stderr, "無法取得目錄 %s 的相關資訊：%s\n", 
				mcopt.chrootdir, strerror(errno));
			abort_makechild(2);
		}else{
			if((chrdirinfo.st_mode & S_IFMT) == S_IFDIR){
				chrdir_oldmode = chrdirinfo.st_mode &
					(S_ISUID |S_ISGID |S_ISVTX | S_IRWXU | S_IRWXG | S_IRWXO);
			}else{
				fprintf(stderr, "%s 並不是目錄\n", mcopt.chrootdir);
				abort_makechild(2);
			}
		}
		if(mcopt.flags & SCTMC_VERBOSE){
			puts("嘗試變更此目錄的權限");
		}
		if(chmod(mcopt.chrootdir, 
			S_IRUSR | S_IXUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) < 0){
			fprintf(stderr, "無法變更目錄 %s 的權限\n", mcopt.chrootdir);
			abort_makechild(2);
		}
	}
	/* 建立用來和 chlid process 溝通的 pipe */
	int childpipe[2], fdinput, fdoutput, fderr, childressize;
	const int fdtablesize = getdtablesize();
	int childmsg[2]; /* [0]存發生事情的名稱 [1]存errno */
	struct rlimit childres;
	int i;
	pid_t pidcopy;
	int readsize = sizeof(int) * 2;
	struct timespec execstart, execfinish;
	struct rusage childrusage;
	int childexitstat;
	char childterminated = 0;
	pipe(childpipe);
	pthread_mutex_lock(&pidmutex);
	pidchild = fork();
	if(pidchild < 0){
		pthread_mutex_unlock(&pidmutex);
		fprintf(stderr, "子程序建立失敗：%s\n", strerror(errno));
		abort_makechild(3);
	}
	else if(pidchild > 0){
		/* 我們原本的 thread */
		pidcopy = pidchild;
		pthread_mutex_unlock(&pidmutex);
		close(childpipe[1]);
		/* 從 pipe 接收 child process 狀態，每次固定要讀 sizeof(int)*2 */
		while(read_size(childpipe[0], (void*)childmsg, readsize) == readsize){
			if(childmsg[0] < 0 || childmsg[0] >= SCTCHILD_MSGMAX){
				/* 會發生嗎？除非被亂七八糟 ptrace 吧 
				 * 這時候先砍掉子程序再說 */
				kill(pidcopy, SIGKILL);
				close(childpipe[0]);
				abort_makechild(4);
			}else{
				if(childmsg[1]){
					fprintf(stderr, "子程序[%d]發生錯誤：%s：%s\n",
							pidcopy, childmsg_text[childmsg[0]], 
							strerror(childmsg[1]));
					kill(pidcopy, SIGKILL);
					close(childpipe[0]);
					abort_makechild(5);
				}else{
					if(mcopt.flags & SCTMC_VERBOSE){
						printf("子程序[%d]：%s\n", pidcopy, 
								childmsg_text[childmsg[0]]);
					}
				}
			}
		}
		clock_gettime(CLOCK_REALTIME, &execstart);
		close(childpipe[0]);

		/* 現在我們確定受測程式已經在執行了，所以需要記錄一下時間
		 * 通知其他 thread，然後就可以把 semaphore 丟掉了 */
		sem_post(&addthr);
		sem_post(&addthr);
		sem_destroy(&addthr);

		/* 開始要 wait 了，莫名其妙 stop 的程式我最多只會送
		 * 5 次 SIGCONT 給你（避免進入無限迴圈） 
		 * 這種奇怪的程式就等著 SLE 吧 */
		for(i=0; i<=5; i++){
			wait4(pidcopy, &childexitstat, WUNTRACED, &childrusage);
			if(WIFSTOPPED(childexitstat) && i!=5){
				if(mcopt.flags & SCTMC_VERBOSE){
					printf("\n子程序因訊號 %d 而暫停，自動傳送 SIGCONT "
							"(第 %d 次)\n",
						WSTOPSIG(childexitstat), i+1);
				}
				kill(pidcopy, SIGCONT);
			}else{
				if(WIFSIGNALED(childexitstat)){
					clock_gettime(CLOCK_REALTIME, &execfinish);
					childterminated = 1;
					judge_signal = WTERMSIG(childexitstat);
					if(judge_signal == SIGXFSZ){
						judge_result = SCTRES_OLE;
					}else{
						judge_result = SCTRES_RE;
					}
					judge_exitcode = WEXITSTATUS(childexitstat);
					break;
				}
				if(WIFEXITED(childexitstat)){
					clock_gettime(CLOCK_REALTIME, &execfinish);
					childterminated = 1;
					judge_result = SCTRES_OK;
					judge_exitcode = WEXITSTATUS(childexitstat);
					break;
				}
			}
		}
		if(!childterminated){
			kill(pidcopy, SIGKILL);
			wait4(pidcopy, &childexitstat, WUNTRACED, &childrusage);
			judge_result = SCTRES_SLE;
			judge_exitcode = WEXITSTATUS(childexitstat);
			clock_gettime(CLOCK_REALTIME, &execfinish);
		}
		pthread_mutex_lock(&judge_tle_mx);
		if(judge_tle && judge_result == SCTRES_RE){
			judge_result = SCTRES_TLE;
		}
		pthread_mutex_unlock(&judge_tle_mx);
		difftimespec(&execstart, &execfinish, &judge_time);
		/* 這裡就要顯示結果了 
		 * XXX 應該拿去 main 放的，用個 struct 包起來？ */
		if(mcopt.flags & SCTMC_VERBOSE){
			putchar('\r');
			puts("========================================");
			fputs("評測結果：", stderr);
			if(judge_result < 0 || judge_result > SCTRES_MAX){
				puts("UD (未定義的結果)");
			}else{
				printf("%s (%s)\n", 
						sctres_text[0][judge_result],
						sctres_text[1][judge_result]);
			}
			printf("執行時間：%ld.%03ld 秒\n",
					judge_time.tv_sec,
					getdvalue_quotient(judge_time.tv_nsec, 1000000));
			printf("程式離開狀態：%d\n", judge_exitcode);
			if(judge_result == SCTRES_RE){
				printf("程式因訊號 %d 而終止\n", judge_signal);
			}
			puts("========================================");
		}else{
			/* 輸出格式：
			 * 第一行是結果代碼、時間、離開狀態 
			 * 第二行是收到的訊號 (只在 RE 時會有) */
			if(judge_result < 0 || judge_result > SCTRES_MAX){
				fputs("UD ", stdout);
			}else{
				fputs(sctres_text[0][judge_result], stdout);
				putchar(' ');
			}
			printf("%ld%03ld %d\n", judge_time.tv_sec,
					getdvalue_quotient(judge_time.tv_nsec, 1000000),
					judge_exitcode);
			if(judge_result == SCTRES_RE){
				printf("%d\n", judge_signal);
			}
		}
	}else{
		/* 子程序 
		 * 由此開始要很小心，因為我們是在 thread 裡面 fork (?) */
		close(childpipe[0]);
		/* close-on-exec 用來關閉這個暫時的資料傳輸通道 */
		fcntl(childpipe[1], F_SETFD, fcntl(childpipe[1],F_GETFD) | FD_CLOEXEC);

		/* 設為獨立的 process group，不過失敗就算了，這不重要 */
		setpgid(getpid(), getpid());

		/* 重設所有 signal handler
		 * XXX 我猜 signal 最大值是 32，還有其他更好的寫法嗎？ */

		fdinput = open(mcopt.inputfile, O_RDONLY);
		if(fdinput < 0){
			SCTCHILD_WRITE_FAILMSG(SCTCHILD_OPEN_INPUT);
			exit(1);
		}else{
			SCTCHILD_WRITE_SUCCMSG(SCTCHILD_OPEN_INPUT);
		}
		if(dup2(fdinput, STDIN_FILENO) < 0){
			SCTCHILD_WRITE_FAILMSG(SCTCHILD_DUP2_INPUT);
			exit(1);
		}else{
			SCTCHILD_WRITE_SUCCMSG(SCTCHILD_DUP2_INPUT);
		}

		/* 注意：開啟輸出檔會直接覆寫現存檔案！ */
		fdoutput = open(mcopt.outputfile, O_WRONLY | O_CREAT | O_TRUNC,
				S_IRUSR | S_IWUSR);
		if(fdoutput < 0){
			SCTCHILD_WRITE_FAILMSG(SCTCHILD_OPEN_OUTPUT);
			exit(1);
		}else{
			SCTCHILD_WRITE_SUCCMSG(SCTCHILD_OPEN_OUTPUT);
		}
		if(dup2(fdoutput, STDOUT_FILENO) < 0){
			SCTCHILD_WRITE_FAILMSG(SCTCHILD_DUP2_OUTPUT);
			exit(1);
		}else{
			SCTCHILD_WRITE_SUCCMSG(SCTCHILD_DUP2_OUTPUT);
		}

		/* 再來就是 stderr 了 */
		if(mcopt.flags & SCTMC_REDIR_STDERR){
			if(dup2(fdoutput, STDERR_FILENO) < 0){
				SCTCHILD_WRITE_FAILMSG(SCTCHILD_DUP2_STDERR);
				exit(1);
			}else{
				SCTCHILD_WRITE_SUCCMSG(SCTCHILD_DUP2_STDERR);
			}
		}else{
			fderr = open(NULL_DEVICE, O_RDONLY);
			if(fderr < 0){
				SCTCHILD_WRITE_FAILMSG(SCTCHILD_OPEN_STDERR);
				exit(1);
			}else{
				SCTCHILD_WRITE_SUCCMSG(SCTCHILD_OPEN_STDERR);
			}
			if(dup2(fderr, STDERR_FILENO) < 0){
				SCTCHILD_WRITE_FAILMSG(SCTCHILD_DUP2_STDERR);
				exit(1);
			}else{
				SCTCHILD_WRITE_SUCCMSG(SCTCHILD_DUP2_STDERR);
			}
		}

		/* 很暴力地把所有不該留著的 fd 關掉 */
		for(i=3; i<fdtablesize; i++){
			if(i != childpipe[1]){
				close(i);
			}
		}
		SCTCHILD_WRITE_SUCCMSG(SCTCHILD_CLOSE_OTHERFD);

		/* 有人要求我要 chroot 嗎？ */
		if(mcopt.chrootdir != NULL){
			if(chroot(mcopt.chrootdir) < 0){
				SCTCHILD_WRITE_FAILMSG(SCTCHILD_CHROOT);
				exit(1);
			}else{
				chdir("/");
				SCTCHILD_WRITE_SUCCMSG(SCTCHILD_CHROOT);
			}
		}


#ifdef HAVE_LINUX_EXTENSIONS
		/* 確保修改身份的時候 capabilities 仍然留著
		 * 等一下再自己把 capabilities 丟掉 */
		prctl(PR_SET_KEEPCAPS, 1, 0, 0, 0);
#endif		

		/* 我要 setgid 嗎？ */
		if(mcopt.flags & SCTMC_SETGID){
			if(setregid(mcopt.gid, mcopt.gid) < 0){
				SCTCHILD_WRITE_FAILMSG(SCTCHILD_SETGID);
				exit(1);
			}else{
				SCTCHILD_WRITE_SUCCMSG(SCTCHILD_SETGID);
			}
			if(setgroups(1, &mcopt.gid) < 0){
				SCTCHILD_WRITE_FAILMSG(SCTCHILD_SETGROUPS);
				exit(1);
			}else{
				SCTCHILD_WRITE_SUCCMSG(SCTCHILD_SETGROUPS);
			}
		}

		/* 我要 setuid 嗎？ */
		if(mcopt.flags & SCTMC_SETUID){
			if(setreuid(mcopt.uid, mcopt.uid) < 0){
				SCTCHILD_WRITE_FAILMSG(SCTCHILD_SETUID);
				exit(1);
			}else{
				SCTCHILD_WRITE_SUCCMSG(SCTCHILD_SETUID);
			}
		}
#ifndef HAVE_LINUX_EXTENSIONS
		else{
			/* 確保 setuid 可執行檔所造成的 effective UID 改變不會傳給子程序 */
			setuid(getuid());
		}
#endif

		/* 開始設定資源限制 */
		if(mcopt.memlimit > 0){
			childressize = mcopt.memlimit * 1024 * 1024;
			childres.rlim_cur = childressize;
			childres.rlim_max = childressize;
			if(setrlimit(RLIMIT_AS, &childres) < 0){
				SCTCHILD_WRITE_FAILMSG(SCTCHILD_SETRLIMIT_VM);
				exit(1);
			}else{
				SCTCHILD_WRITE_SUCCMSG(SCTCHILD_SETRLIMIT_VM);
			}
		}

		if(mcopt.outlimit > 0){
			childressize = mcopt.outlimit * 1024 * 1024;
			childres.rlim_cur = childressize;
			childres.rlim_max = childressize;
			if(setrlimit(RLIMIT_FSIZE, &childres) < 0){
				SCTCHILD_WRITE_FAILMSG(SCTCHILD_SETRLIMIT_FSIZE);
				exit(1);
			}else{
				SCTCHILD_WRITE_SUCCMSG(SCTCHILD_SETRLIMIT_FSIZE);
			}
		}

		/* 從現在開始，不准再開檔案了 */
		childres.rlim_cur = 0;
		childres.rlim_max = 0;
		if(setrlimit(
#ifdef RLIMIT_NOFILE
					RLIMIT_NOFILE
#else
					RLIMIT_OFILE
#endif
					, &childres) < 0){
			SCTCHILD_WRITE_FAILMSG(SCTCHILD_SETRLIMIT_OPENFILE);
			exit(1);
		}else{
			SCTCHILD_WRITE_SUCCMSG(SCTCHILD_SETRLIMIT_OPENFILE);
		}

		/* 從現在開始，不可以再產生新程序了 */
		if(setrlimit(RLIMIT_NPROC, &childres) < 0){
			SCTCHILD_WRITE_FAILMSG(SCTCHILD_SETRLIMIT_PROC);
			exit(1);
		}else{
			SCTCHILD_WRITE_SUCCMSG(SCTCHILD_SETRLIMIT_PROC);
		}
		

		/* 非 Linux 系統就到此結束了，可以 exec 了 */

#ifdef HAVE_LINUX_EXTENSIONS
		/* 正在丟掉 capabilities */
		cap_t nocap = cap_init();
		cap_clear_flag(nocap, CAP_EFFECTIVE);
		cap_clear_flag(nocap, CAP_INHERITABLE);
		cap_clear_flag(nocap, CAP_PERMITTED);
		if(cap_set_proc(nocap) < 0){
			SCTCHILD_WRITE_FAILMSG(SCTCHILD_CAPSET);
			exit(1);
		}else{
			SCTCHILD_WRITE_SUCCMSG(SCTCHILD_CAPSET);
		}
		/* 需要 drop bound 嗎？再考慮看看 */
#endif

		if(mcopt.chrootdir == NULL){
			execl(mcopt.executable, mcopt.executable, NULL);
		}else{
			if(mcopt.flags & SCTMC_NOCOPY){
				execl(mcopt.executable, mcopt.executable, NULL);
			}else{
				execl(execshortname, execshortname, NULL);
			}
		}

		SCTCHILD_WRITE_FAILMSG(SCTCHILD_EXEC);
		exit(2);
	}
	/* Judge 完成，準備離開 */
	sctjudge_makechild_cleanup(&mcopt, chrdir_oldmode, execdestname);
	(*(int*)toreturn) = 0;
	return toreturn;
}
